﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Runtime.InteropServices;
using ComfileTech.ComfilePi.IO;
using System.Collections.ObjectModel;

namespace ComfileTech.ComfilePi.IO.Test
{
    public class State<T>
    {
        public State(T value)
        {
            Value = value;
        }

        public T Value
        { get; set; }
    }

    public class DeviceServer : IDisposable
    {
        public DeviceServer()
        {
            _device = new CP_IO22_A4_2();

            _digitalInputs = new Dictionary<Pin, State<bool>>();
            foreach(var i in _device.DigitalInputs)
            {
                _digitalInputs.Add(i.Key, new State<bool>(i.Value.State));
            }

            _digitalOutputs = new Dictionary<Pin, State<bool>>();
            foreach(var o in _device.DigitalOutputs)
            {
                _digitalOutputs.Add(o.Key, new State<bool>(o.Value.State));
            }

            _digitalThread = new Thread(SyncDigital);
            _digitalThread.IsBackground = true;
            _digitalThread.Priority = ThreadPriority.Highest;
            _digitalThread.Start();

            _analogInputs = new Dictionary<AnalogInput.ChannelID, State<ushort>>();
            foreach(var i in _device.AnalogInputs)
            {
                _analogInputs.Add(i.Key, new State<ushort>(i.Value.Read()));
            }

            _analogOutputs = new Dictionary<AnalogOutput.ChannelID, State<ushort>>();
            foreach (var i in _device.AnalogOutputs)
            {
                _analogOutputs.Add(i.Key, new State<ushort>(i.Value.State));
            }

            _analogThread = new Thread(SyncAnalog);
            _analogThread.IsBackground = true;
            _analogThread.Priority = ThreadPriority.Highest;
            _analogThread.Start();
        }

        CP_IO22_A4_2 _device;

        public event Action Changed;

        Dictionary<Pin, State<bool>> _digitalInputs;
        public IReadOnlyDictionary<Pin, State<bool>> DigitalInputs => _digitalInputs;


        Dictionary<Pin, State<bool>> _digitalOutputs;
        public IReadOnlyDictionary<Pin, State<bool>> DigitalOutputs => _digitalOutputs;


        Dictionary<AnalogInput.ChannelID, State<ushort>> _analogInputs;
        public IReadOnlyDictionary<AnalogInput.ChannelID, State<ushort>> AnalogInputs => _analogInputs;

        Dictionary<AnalogOutput.ChannelID, State<ushort>> _analogOutputs;
        public IReadOnlyDictionary<AnalogOutput.ChannelID, State<ushort>> AnalogOutputs => _analogOutputs;


        Thread _digitalThread;
        Thread _analogThread;
        volatile bool _stopThread;

        void SyncDigital()
        {
            var digitalInputs = DigitalInputs.Values.Select(i => i.Value).ToArray();
            var digitalOutputs = DigitalOutputs.Values.Select(o => o.Value).ToArray();

            while (!_stopThread)
            {
                foreach(var i in _device.DigitalInputs)
                {
                    _digitalInputs[i.Key].Value = i.Value.State;
                }

                foreach (var o in _device.DigitalOutputs)
                {
                    o.Value.State = _digitalOutputs[o.Key].Value;
                }

                var newDigitalInputs = DigitalInputs.Values.Select(i => i.Value).ToArray();
                var newDigitalOutputs = DigitalOutputs.Values.Select(i => i.Value).ToArray();

                if (!newDigitalInputs.SequenceEqual(digitalInputs) 
                    || !newDigitalOutputs.SequenceEqual(digitalOutputs))
                {
                    Changed?.Invoke();

                    digitalInputs = newDigitalInputs;
                    digitalOutputs = newDigitalOutputs;
                }
            }
        }

        void SyncAnalog()
        {
            var analogInputs = AnalogInputs.Values.Select(i => i.Value).ToArray();
            var analogOutputs = AnalogOutputs.Values.Select(o => o.Value).ToArray();

            while (!_stopThread)
            {
                foreach (var i in _device.AnalogInputs)
                {
                    i.Value.ScheduleRead();

                    // Set analog outputs while waiting for analog input;
                    // should be more efficient this way
                    while (!i.Value.IsReady())
                    {
                        foreach (var o in _device.AnalogOutputs)
                        {
                            o.Value.State = _analogOutputs[o.Key].Value;
                        }

                        var newAnalogOutputs = AnalogOutputs.Values.Select(o => o.Value).ToArray();

                        if (!newAnalogOutputs.SequenceEqual(analogOutputs))
                        {
                            Changed?.Invoke();

                            analogOutputs = newAnalogOutputs;
                        }
                    }

                    _analogInputs[i.Key].Value = i.Value.ReadValue();

                    var newAnalogInputs = AnalogInputs.Values.Select(a => a.Value).ToArray();

                    if (!newAnalogInputs.SequenceEqual(analogInputs))
                    {
                        Changed?.Invoke();

                        analogInputs = newAnalogInputs;
                    }
                }
            }
        }

        public void Dispose()
        {
            Console.WriteLine("Disposing Device Server");

            if (_digitalThread != null)
            {
                _stopThread = true;
                _digitalThread.Join();
                _analogThread.Join();
            }

            _digitalThread = null;
            _analogThread = null;

            if (_device != null)
            {
                _device.Dispose();
            }

            _device = null;
        }
    }
}
